*! x13as v5.2 - Simplified version
*! Stata wrapper for Census X-13ARIMA-SEATS seasonal adjustment  
*! Simplified to 4 spec options: SEATS default, X11+auto, X11+none, X11+log
*! Author: Jed Kolko
*! Date: 20 Nov 2025
*! change in v5.1: the noadmiss=yes option for SEATS, and automatic outlier detection
*! change in v5.2: fixed transform block to correctly pass "auto" to spec file


capture program drop x13as
program define x13as, rclass
    version 14.0
    syntax varname(ts) [if] [in], ///
        [ ///
        Method(string)          /// seats or x11 (default: seats)
        TRAnsform(string)       /// auto/none/log (for X-11 only, default: auto)
        SAving(name)            /// name for seasonally adjusted series
        TRend(name)             /// name for trend component
        IRRegular(name)         /// name for irregular component  
        SFactor(name)           /// name for seasonal factors
        WORKdir(string)         /// working directory for temp files
        KEEPfiles               /// keep temporary files
        REPlace                 /// replace existing variables
        DETails                 /// show details of execution
        ]

    marksample touse
    qui count if `touse'
    if r(N) == 0 error 2000

    * Locate X-13 executable
    local x13exe "./x13as.exe"

    local origvar "`varlist'"
    local Nobs = r(N)

    * Check tsset
    qui tsset
    local timevar  "`r(timevar)'"
    local panelvar "`r(panelvar)'"
    if "`timevar'" == "" {
        di as error "Data must be tsset"
        exit 459
    }
    if "`panelvar'" != "" {
        di as error "Panel data not supported"
        exit 459
    }

    local tsfmt "`r(unit1)'"

    * Detect frequency
    if "`tsfmt'" == "m"      local period = 12
    else if "`tsfmt'" == "q" local period = 4
    else {
        di as error "Data must be monthly or quarterly"
        exit 459
    }

    * Set defaults
    if "`method'" == "" local method "seats"
    if "`transform'" == "" local transform "auto"
    if "`saving'" == "" local saving "`origvar'_sa"

    * Validate inputs
    if !inlist("`method'", "seats", "x11") {
        di as error "method() must be 'seats' or 'x11'"
        exit 198
    }
    if !inlist("`transform'", "auto", "none", "log") {
        di as error "transform() must be 'auto', 'none', or 'log'"
        exit 198
    }

    * Check for variable conflicts
    if "`replace'" == "" {
        capture confirm variable `saving'
        if !_rc {
            di as error "Variable `saving' already exists. Use replace option."
            exit 110
        }
        if "`trend'" != "" {
            capture confirm variable `trend'
            if !_rc {
                di as error "Variable `trend' already exists. Use replace option."
                exit 110
            }
        }
        if "`irregular'" != "" {
            capture confirm variable `irregular'
            if !_rc {
                di as error "Variable `irregular' already exists. Use replace option."
                exit 110
            }
        }
        if "`sfactor'" != "" {
            capture confirm variable `sfactor'
            if !_rc {
                di as error "Variable `sfactor' already exists. Use replace option."
                exit 110
            }
        }
    }

    * Setup working directory
    if "`workdir'" == "" {
        local workdir "`c(tmpdir)'"
        local workdir = regexr("`workdir'", "[\\/]$", "")
    }
    capture mkdir "`workdir'"

    * Generate unique filenames
    local safevar = ustrregexra("`origvar'", "[^A-Za-z0-9_-]", "_")
    local timestamp = floor(runiform()*1000000)
    local fileprefix "x13_`safevar'_`timestamp'"

    if "`c(os)'" == "Windows" local sep "\"
    else local sep "/"

    local datafile "`workdir'`sep'`fileprefix'.dat"
    local specfile "`workdir'`sep'`fileprefix'.spc"
    local specbase = subinstr("`specfile'", ".spc", "", .)
    local outfile  "`workdir'`sep'`fileprefix'.out"

    * Export data to X-13 format
    preserve
    qui keep if `touse'
    qui keep `timevar' `origvar'
    qui drop if missing(`origvar')
    qui count
    if r(N) == 0 {
        di as error "No observations after dropping missing values"
        restore
        exit 2000
    }
    
    sum `timevar', meanonly
    local startdate = r(min)
    if `period' == 12 {
        local startyear   = year(dofm(`startdate'))
        local startperiod = month(dofm(`startdate'))
        local startperiod_fmt = string(`startperiod', "%02.0f")
    }
    else {
        local startyear   = year(dofq(`startdate'))
        local startperiod = quarter(dofq(`startdate'))
        local startperiod_fmt = string(`startperiod', "%1.0f")
    }

    * Write data file
    tempname fh
    file open `fh' using "`datafile'", write replace
    qui count
    forval i = 1/`r(N)' {
        local v = `origvar'[`i']
        file write `fh' "`v'" _n
    }
    file close `fh'
    restore

    * Create X-13 specification file
    tempname fhs
    file open `fhs' using "`specfile'", write replace
    
    * Convert path for X-13 (use forward slashes)
    local datafile_unix = subinstr("`datafile'", "\", "/", .)
    
    * Series block
    file write `fhs' "series{" _n
    file write `fhs' "    file = " _char(34) "`datafile_unix'" _char(34) _n
    file write `fhs' "    format = " _char(34) "free" _char(34) _n
    file write `fhs' "    period = `period'" _n
    file write `fhs' "    start = `startyear'.`startperiod_fmt'" _n
    file write `fhs' "}" _n _n
    
    * Transform block
	/*
    if "`transform'" != "auto" {
        file write `fhs' "transform{" _n
        file write `fhs' "    function = `transform'" _n
        file write `fhs' "}" _n _n
    }
	*/
	* Transform block - write it regardless of value
file write `fhs' "transform{" _n
file write `fhs' "    function = `transform'" _n
file write `fhs' "}" _n _n


    
    * Regression block (trading day and Easter)
    file write `fhs' "regression{" _n
    file write `fhs' "    aictest = (td easter)" _n
    file write `fhs' "}" _n _n
    
    * ARIMA block (automatic model selection)
    file write `fhs' "automdl{" _n
    file write `fhs' "}" _n _n
    
	* Outlier detection (matches R seasonal package)
file write `fhs' "outlier{" _n
file write `fhs' "}" _n _n
	
    * Decomposition block
    if "`method'" == "seats" {
        file write `fhs' "seats{" _n
        file write `fhs' "    noadmiss = yes" _n
        file write `fhs' "    save = (s11 s12 s13 s16)" _n
        file write `fhs' "}" _n _n
    }
    else {
        file write `fhs' "x11{" _n
        file write `fhs' "    save = (d11 d12 d13 d16)" _n
        file write `fhs' "}" _n _n
    }
    
    file close `fhs'

    * Show details if requested
    if "`details'" != "" {
        di _n as text "X-13 Specification File:"
        di as text "{hline 60}"
        type "`specfile'"
        di as text "{hline 60}" _n
        di as text "Running X-13ARIMA-SEATS..."
        shell "`x13exe'" "`specbase'"
    }
    else {
        qui shell "`x13exe'" "`specbase'"
    }

    * Determine output file names
    if "`method'" == "seats" {
        local safile    "`workdir'`sep'`fileprefix'.s11"
        local trendfile "`workdir'`sep'`fileprefix'.s12"
        local irregfile "`workdir'`sep'`fileprefix'.s13"
        local sffile    "`workdir'`sep'`fileprefix'.s16"
    }
    else {
        local safile    "`workdir'`sep'`fileprefix'.d11"
        local trendfile "`workdir'`sep'`fileprefix'.d12"
        local irregfile "`workdir'`sep'`fileprefix'.d13"
        local sffile    "`workdir'`sep'`fileprefix'.d16"
    }

    * Check if X-13 succeeded
    capture confirm file "`safile'"
    if _rc {
        di as error "X-13 execution failed - no output file created"
        di as error "Try details keepfiles options to diagnose"
        exit 601
    }

    * Import seasonally adjusted series
    quietly __x13_import_one using "`safile'", ///
        freq(`period') outvar(`saving') timevar(`timevar')
    quietly replace `saving' = . if !`touse'
    label var `saving' "Seasonally adjusted `origvar'"

    * Import trend if requested
    if "`trend'" != "" {
        capture confirm file "`trendfile'"
        if !_rc {
            quietly __x13_import_one using "`trendfile'", ///
                freq(`period') outvar(__trend__) timevar(`timevar')
            capture drop `trend'
            rename __trend__ `trend'
            label var `trend' "Trend-cycle of `origvar'"
            quietly replace `trend' = . if !`touse'
        }
    }

    * Import irregular if requested
    if "`irregular'" != "" {
        capture confirm file "`irregfile'"
        if !_rc {
            quietly __x13_import_one using "`irregfile'", ///
                freq(`period') outvar(__ir__) timevar(`timevar')
            capture drop `irregular'
            rename __ir__ `irregular'
            label var `irregular' "Irregular component of `origvar'"
            quietly replace `irregular' = . if !`touse'
        }
    }

    * Import seasonal factors if requested
    if "`sfactor'" != "" {
        capture confirm file "`sffile'"
        if !_rc {
            quietly __x13_import_one using "`sffile'", ///
                freq(`period') outvar(__sf__) timevar(`timevar')
            capture drop `sfactor'
            rename __sf__ `sfactor'
            label var `sfactor' "Seasonal factors of `origvar'"
            quietly replace `sfactor' = . if !`touse'
        }
    }

    * Cleanup temporary files
    if "`keepfiles'" == "" {
        capture erase "`datafile'"
        capture erase "`specfile'"
        capture erase "`outfile'"
        capture erase "`safile'"
        capture erase "`trendfile'"
        capture erase "`irregfile'"
        capture erase "`sffile'"
        capture erase "`workdir'`sep'`fileprefix'.err"
        capture erase "`workdir'`sep'`fileprefix'.log"
    }
    else {
        di as text _n "Files saved in: `workdir'"
        di as text "Prefix: `fileprefix'"
    }

    * Display results
    di _n as text "X-13ARIMA-SEATS completed successfully"
    di as text "Method: " as result "`method'" ///
        as text " | Transform: " as result "`transform'"
    di as text "Input: " as result "`origvar'" ///
        as text " -> Output: " as result "`saving'"

    * Return results
    return local method     "`method'"
    return local transform  "`transform'"
    return local saving     "`saving'"
    if "`trend'" != ""     return local trend     "`trend'"
    if "`irregular'" != "" return local irregular "`irregular'"
    if "`sfactor'" != ""   return local sfactor   "`sfactor'"
    return local workdir    "`workdir'"
    return local fileprefix "`fileprefix'"
    return scalar N         = `Nobs'

end


* Helper program to import X-13 output files
capture program drop __x13_import_one
program define __x13_import_one
    version 14.0
    syntax using/, Freq(integer) OUTVAR(name) TIMEVAR(name)

    preserve
    quietly import delimited "`using'", clear delimiter(whitespace) ///
        varnames(1) stringcols(_all)
    quietly ds
    local vlist `r(varlist)'
    local dcol : word 1 of `vlist'
    local vcol : word 2 of `vlist'

    gen double __date = .
    gen double __val  = .

    quietly count
    forval i = 1/`r(N)' {
        local ds = strtrim(`dcol'[`i'])
        local vs = strtrim(`vcol'[`i'])
        capture confirm number `vs'
        if !_rc {
            capture confirm number `ds'
            if !_rc {
                if `freq' == 12 {
                    local yr  = substr("`ds'", 1, 4)
                    local per = substr("`ds'", 5, .)
                    quietly replace __date = ym(`yr', `per') in `i'
                }
                else {
                    local yr  = substr("`ds'", 1, 4)
                    local per = substr("`ds'", 5, .)
                    quietly replace __date = yq(`yr', `per') in `i'
                }
                quietly replace __val = real("`vs'") in `i'
            }
        }
    }
    quietly drop if missing(__date)
    keep __date __val
    rename __date `timevar'
    tempfile __res
    quietly save "`__res'"
    restore
    
    quietly merge 1:1 `timevar' using "`__res'", nogen keep(master match)
    capture drop `outvar'
    gen double `outvar' = __val
    drop __val
end
